/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.svl;

import static org.apache.juneau.commons.lang.StateEnum.*;
import static org.apache.juneau.commons.reflect.ReflectionUtils.*;
import static org.apache.juneau.commons.utils.CollectionUtils.*;
import static org.apache.juneau.commons.utils.StringUtils.*;
import static org.apache.juneau.commons.utils.ThrowableUtils.*;
import static org.apache.juneau.commons.utils.Utils.*;

import java.io.*;
import java.lang.reflect.*;
import java.util.*;

import org.apache.juneau.commons.collections.*;
import org.apache.juneau.commons.lang.*;
import org.apache.juneau.cp.*;

/**
 * A var resolver session that combines a {@link VarResolver} with one or more session objects.
 *
 * <p>
 * Instances of this class are considered light-weight and fast to construct, use, and discard.
 *
 * <p>
 * This class contains the workhorse code for var resolution.
 *
 * <p>
 * Instances of this class are created through the {@link VarResolver#createSession()} and
 * {@link VarResolver#createSession(BeanStore)} methods.
 *
 * <h5 class='section'>Notes:</h5><ul>
 * 	<li class='warn'>This class is not guaranteed to be thread safe.
 * </ul>
 *
 * <h5 class='section'>See Also:</h5><ul>
 * 	<li class='link'><a class="doclink" href="https://juneau.apache.org/docs/topics/SimpleVariableLanguageBasics">Simple Variable Language Basics</a>
 * </ul>
 */
@SuppressWarnings("resource")
public class VarResolverSession {

	private static final AsciiSet AS1 = AsciiSet.of("\\{"), AS2 = AsciiSet.of("\\${}");

	private static boolean containsVars(Collection<?> c) {
		var f = Flag.create();
		c.forEach(x -> {
			if (x instanceof CharSequence && x.toString().contains("$"))
				f.set();
		});
		return f.isSet();
	}

	private static boolean containsVars(Map<?,?> m) {
		var f = Flag.create();
		m.forEach((k, v) -> {
			if (v instanceof CharSequence && v.toString().contains("$"))
				f.set();
		});
		return f.isSet();
	}

	private static boolean containsVars(Object array) {
		for (var i = 0; i < Array.getLength(array); i++) {
			var o = Array.get(array, i);
			if (o instanceof CharSequence && o.toString().contains("$"))
				return true;
		}
		return false;
	}

	/*
	 * Checks to see if string is of the simple form "$X{...}" with no embedded variables.
	 * This is a common case, and we can avoid using StringWriters.
	 */
	private static boolean isSimpleVar(String s) {
		// S1: Not in variable, looking for $
		// S2: Found $, Looking for {
		// S3: Found {, Looking for }
		// S4: Found }

		int length = s.length();
		var state = S1;
		for (var i = 0; i < length; i++) {
			var c = s.charAt(i);
			if (state == S1) {
				if (c == '$') {
					state = S2;
				} else {
					return false;
				}
			} else if (state == S2) {
				if (c == '{') {
					state = S3;
				} else if (c < 'A' || c > 'z' || (c > 'Z' && c < 'a')) {   // False trigger "$X "
					return false;
				}
			} else if (state == S3) {
				if (c == '}')
					state = S4;
				else if (c == '{' || c == '$')
					return false;
			} else if (state == S4) {
				return false;
			}
		}
		return state == S4;
	}

	private final VarResolver context;

	private final BeanStore beanStore;

	/**
	 * Constructor.
	 *
	 * @param context
	 * 	The {@link VarResolver} context object that contains the {@link Var Vars} and context objects associated with
	 * 	that resolver.
	 * @param beanStore The bean store to use for resolving beans needed by vars.
	 *
	 */
	public VarResolverSession(VarResolver context, BeanStore beanStore) {
		this.context = context;
		this.beanStore = BeanStore.of(beanStore);
	}

	/**
	 * Adds a bean to this session.
	 *
	 * @param <T> The bean type.
	 * @param c The bean type.
	 * @param value The bean.
	 * @return This object.
	 */
	public <T> VarResolverSession bean(Class<T> c, T value) {
		beanStore.addBean(c, value);
		return this;
	}

	/**
	 * Returns the bean from the registered bean store.
	 *
	 * @param <T> The value type.
	 * @param c The bean type.
	 * @return
	 * 	The bean.
	 * 	<br>Never <jk>null</jk>.
	 */
	public <T> Optional<T> getBean(Class<T> c) {
		Optional<T> t = beanStore.getBean(c);
		if (! t.isPresent())
			t = context.beanStore.getBean(c);
		return t;
	}

	/**
	 * Resolve all variables in the specified string.
	 *
	 * @param s
	 * 	The string to resolve variables in.
	 * @return
	 * 	The new string with all variables resolved, or the same string if no variables were found.
	 * 	<br>Returns <jk>null</jk> if the input was <jk>null</jk>.
	 */
	public String resolve(String s) {

		if (s == null || s.isEmpty() || (s.indexOf('$') == -1 && s.indexOf('\\') == -1))
			return s;

		// Special case where value consists of a single variable with no embedded variables (e.g. "$X{...}").
		// This is a common case, so we want an optimized solution that doesn't involve string builders.
		if (isSimpleVar(s)) {
			String var = s.substring(1, s.indexOf('{'));
			String val = s.substring(s.indexOf('{') + 1, s.length() - 1);
			Var v = getVar(var);
			if (nn(v)) {
				try {
					if (v.streamed) {
						var sw = new StringWriter();
						v.resolveTo(this, sw, val);
						return sw.toString();
					}
					s = v.doResolve(this, val);
					if (s == null)
						s = "";
					return (v.allowRecurse() ? resolve(s) : s);
				} catch (VarResolverException e) {
					throw e;
				} catch (Exception e) {
					throw new VarResolverException(e, "Problem occurred resolving variable ''{0}'' in string ''{1}''", var, s);
				}
			}
			return s;
		}

		try {
			return resolveTo(s, new StringWriter()).toString();
		} catch (IOException e) {
			throw toRex(e); // Never happens.
		}
	}

	/**
	 * Resolves the specified strings in the string array.
	 *
	 * @param in The string array containing variables to resolve.
	 * @return An array with resolved strings.
	 */
	public String[] resolve(String[] in) {
		var out = new String[in.length];
		for (var i = 0; i < in.length; i++)
			out[i] = resolve(in[i]);
		return out;
	}

	/**
	 * Convenience method for resolving variables in arbitrary objects.
	 *
	 * <p>
	 * Supports resolving variables in the following object types:
	 * <ul>
	 * 	<li>{@link CharSequence}
	 * 	<li>Arrays containing values of type {@link CharSequence}.
	 * 	<li>Collections containing values of type {@link CharSequence}.
	 * 		<br>Collection class must have a no-arg constructor.
	 * 	<li>Maps containing values of type {@link CharSequence}.
	 * 		<br>Map class must have a no-arg constructor.
	 * </ul>
	 *
	 * @param <T> The value type.
	 * @param o The object.
	 * @return The same object if no resolution was needed, otherwise a new object or data structure if resolution was
	 * needed.
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	public <T> T resolve(T o) {
		if (o == null)
			return null;
		if (o instanceof CharSequence o2)
			return (T)resolve(o2.toString());
		if (isArray(o)) {
			if (! containsVars(o))
				return o;
			var o2 = Array.newInstance(o.getClass().getComponentType(), Array.getLength(o));
			for (var i = 0; i < Array.getLength(o); i++)
				Array.set(o2, i, resolve(Array.get(o, i)));
			return (T)o2;
		}
		if (o instanceof Set o2) {
			try {
				if (! containsVars(o2))
					return o;
				Set o3 = info(o).getDeclaredConstructor(x -> x.isPublic() && x.getParameterCount() == 0).map(ci -> safe(() -> (Set)ci.inner().newInstance())).orElseGet(LinkedHashSet::new);
				Set o4 = o3;
				o2.forEach(x -> o4.add(resolve(x)));
				return (T)o3;
			} catch (VarResolverException e) {
				throw e;
			} catch (Exception e) {
				throw new VarResolverException(e, "Problem occurred resolving set.");
			}
		}
		if (o instanceof List o2) {
			try {
				if (! containsVars(o2))
					return o;
				List o3 = info(o).getDeclaredConstructor(x -> x.isPublic() && x.getParameterCount() == 0).map(ci -> safe(() -> (List)ci.inner().newInstance())).orElseGet(() -> list());
				List o4 = o3;
				o2.forEach(x -> o4.add(resolve(x)));
				return (T)o3;
			} catch (VarResolverException e) {
				throw e;
			} catch (Exception e) {
				throw new VarResolverException(e, "Problem occurred resolving collection.");
			}
		}
		if (o instanceof Map o2) {
			try {
				if (! containsVars(o2))
					return o;
				Map o3 = info(o).getDeclaredConstructor(x -> x.isPublic() && x.getParameterCount() == 0).map(ci -> safe(() -> (Map)ci.inner().newInstance())).orElseGet(LinkedHashMap::new);
				Map o4 = o3;
				o2.forEach((k, v) -> o4.put(k, resolve(v)));
				return (T)o3;
			} catch (VarResolverException e) {
				throw e;
			} catch (Exception e) {
				throw new VarResolverException(e, "Problem occurred resolving map.");
			}
		}
		return o;
	}

	/**
	 * Resolves variables in the specified string and sends the output to the specified writer.
	 *
	 * <p>
	 * More efficient than first parsing to a string and then serializing to the writer since this method doesn't need
	 * to construct a large string.
	 *
	 * @param s The string to resolve variables in.
	 * @param out The writer to write to.
	 * @return The same writer.
	 * @throws IOException Thrown by underlying stream.
	 */
	public Writer resolveTo(String s, Writer out) throws IOException {

		// S1: Not in variable, looking for $
		// S2: Found $, Looking for {
		// S3: Found {, Looking for }

		var state = S1;
		var isInEscape = false;
		var hasInternalVar = false;
		var hasInnerEscapes = false;
		var varType = (String)null;
		var varVal = (String)null;
		var x = 0;
		var x2 = 0;
		var depth = 0;
		var length = s.length();
		for (var i = 0; i < length; i++) {
			var c = s.charAt(i);
			if (state == S1) {
				if (isInEscape) {
					if (c == '\\' || c == '$') {
						out.append(c);
					} else {
						out.append('\\').append(c);
					}
					isInEscape = false;
				} else if (c == '\\') {
					isInEscape = true;
				} else if (c == '$') {
					x = i;
					x2 = i;
					state = S2;
				} else {
					out.append(c);
				}
			} else if (state == S2) {
				if (isInEscape) {
					isInEscape = false;
				} else if (c == '\\') {
					hasInnerEscapes = true;
					isInEscape = true;
				} else if (c == '{') {
					varType = s.substring(x + 1, i);
					x = i;
					state = S3;
				} else if (c < 'A' || c > 'z' || (c > 'Z' && c < 'a')) {  // False trigger "$X "
					if (hasInnerEscapes)
						out.append(unescapeChars(s.substring(x, i + 1), AS1));
					else
						out.append(s, x, i + 1);
					x = i + 1;
					state = S1;
					hasInnerEscapes = false;
				}
			} else if (state == S3) {
				if (isInEscape) {
					isInEscape = false;
				} else if (c == '\\') {
					isInEscape = true;
					hasInnerEscapes = true;
				} else if (c == '{') {
					depth++;
					hasInternalVar = true;
				} else if (c == '}') {
					if (depth > 0) {
						depth--;
					} else {
						varVal = s.substring(x + 1, i);
						Var r = getVar(varType);
						if (r == null) {
							if (hasInnerEscapes)
								out.append(unescapeChars(s.substring(x2, i + 1), AS2));
							else
								out.append(s, x2, i + 1);
							x = i + 1;
						} else {
							varVal = (hasInternalVar && r.allowNested() ? resolve(varVal) : varVal);
							try {
								if (r.streamed)
									r.resolveTo(this, out, varVal);
								else {
									String replacement = r.doResolve(this, varVal);
									if (replacement == null)
										replacement = "";
									// If the replacement also contains variables, replace them now.
									if (replacement.indexOf('$') != -1 && r.allowRecurse())
										replacement = resolve(replacement);
									out.append(replacement);
								}
							} catch (VarResolverException e) {
								throw e;
							} catch (Exception e) {
								throw new VarResolverException(e, "Problem occurred resolving variable ''{0}'' in string ''{1}''", varType, s);
							}
							x = i + 1;
						}
						state = S1;
						hasInnerEscapes = false;
					}
				}
			}
		}
		if (isInEscape)
			out.append('\\');
		else if (state == S2)
			out.append('$').append(unescapeChars(s.substring(x + 1), AS1));
		else if (state == S3)
			out.append('$').append(varType).append('{').append(unescapeChars(s.substring(x + 1), AS2));
		return out;
	}

	protected FluentMap<String,Object> properties() {
		// @formatter:off
		return filteredBeanPropertyMap()
			.a("context.beanStore", this.context.beanStore)
			.a("var", this.context.getVarMap().keySet())
			.a("session.beanStore", beanStore);
		// @formatter:on
	}

	@Override /* Overridden from Object */
	public String toString() {
		return r(properties());
	}

	/**
	 * Returns the {@link Var} with the specified name.
	 *
	 * @param name The var name (e.g. <js>"S"</js>).
	 * @return The {@link Var} instance, or <jk>null</jk> if no <c>Var</c> is associated with the specified name.
	 */
	protected Var getVar(String name) {
		Var v = this.context.getVarMap().get(name);
		return nn(v) && v.canResolve(this) ? v : null;
	}
}