/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.retries.internal.ratelimiter;

import java.time.Duration;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.retries.internal.ratelimiter.RateLimiterAcquireResponse;
import software.amazon.awssdk.retries.internal.ratelimiter.RateLimiterClock;
import software.amazon.awssdk.retries.internal.ratelimiter.RateLimiterUpdateResponse;

@SdkInternalApi
public class RateLimiterTokenBucket {
    private final AtomicReference<PersistentState> stateReference;
    private final RateLimiterClock clock;

    RateLimiterTokenBucket(RateLimiterClock clock) {
        this.clock = clock;
        this.stateReference = new AtomicReference<PersistentState>(new PersistentState());
    }

    public RateLimiterAcquireResponse tryAcquire() {
        StateUpdate<Duration> update = this.updateState(ts -> ts.tokenBucketAcquire(this.clock, 1.0));
        return RateLimiterAcquireResponse.create((Duration)((StateUpdate)update).result);
    }

    public RateLimiterUpdateResponse updateRateAfterThrottling() {
        StateUpdate<Void> update = this.consumeState(ts -> ts.updateClientSendingRate(this.clock, true));
        return RateLimiterUpdateResponse.builder().measuredTxRate(((StateUpdate)update).newState.measuredTxRate()).fillRate(((StateUpdate)update).newState.fillRate()).build();
    }

    public RateLimiterUpdateResponse updateRateAfterSuccess() {
        StateUpdate<Void> update = this.consumeState(ts -> ts.updateClientSendingRate(this.clock, false));
        return RateLimiterUpdateResponse.builder().measuredTxRate(((StateUpdate)update).newState.measuredTxRate()).fillRate(((StateUpdate)update).newState.fillRate()).build();
    }

    private StateUpdate<Void> consumeState(Consumer<TransientState> mutator) {
        return this.updateState(ts -> {
            mutator.accept((TransientState)ts);
            return null;
        });
    }

    private <T> StateUpdate<T> updateState(Function<TransientState, T> mutator) {
        T result;
        TransientState transientState;
        PersistentState updated;
        PersistentState current;
        do {
            current = this.stateReference.get();
            transientState = current.toTransient();
            result = mutator.apply(transientState);
        } while (!this.stateReference.compareAndSet(current, updated = transientState.toPersistent()));
        return new StateUpdate<T>(updated, result);
    }

    static final class PersistentState {
        private final double fillRate;
        private final double maxCapacity;
        private final double currentCapacity;
        private final boolean lastTimestampIsSet;
        private final double lastTimestamp;
        private final boolean enabled;
        private final double measuredTxRate;
        private final double lastTxRateBucket;
        private final long requestCount;
        private final double lastMaxRate;
        private final double lastThrottleTime;
        private final double timeWindow;
        private final double newTokenBucketRate;

        private PersistentState() {
            this.fillRate = 0.0;
            this.maxCapacity = 0.0;
            this.currentCapacity = 0.0;
            this.lastTimestampIsSet = false;
            this.lastTimestamp = 0.0;
            this.enabled = false;
            this.measuredTxRate = 0.0;
            this.lastTxRateBucket = 0.0;
            this.requestCount = 0L;
            this.lastMaxRate = 0.0;
            this.lastThrottleTime = 0.0;
            this.timeWindow = 0.0;
            this.newTokenBucketRate = 0.0;
        }

        PersistentState(TransientState state) {
            this.fillRate = state.fillRate;
            this.maxCapacity = state.maxCapacity;
            this.currentCapacity = state.currentCapacity;
            this.lastTimestampIsSet = state.lastTimestampIsSet;
            this.lastTimestamp = state.lastTimestamp;
            this.enabled = state.enabled;
            this.measuredTxRate = state.measuredTxRate;
            this.lastTxRateBucket = state.lastTxRateBucket;
            this.requestCount = state.requestCount;
            this.lastMaxRate = state.lastMaxRate;
            this.lastThrottleTime = state.lastThrottleTime;
            this.timeWindow = state.timeWindow;
            this.newTokenBucketRate = state.newTokenBucketRate;
        }

        TransientState toTransient() {
            return new TransientState(this);
        }

        public double fillRate() {
            return this.fillRate;
        }

        public double measuredTxRate() {
            return this.measuredTxRate;
        }
    }

    static final class TransientState {
        private static final double MIN_FILL_RATE = 0.5;
        private static final double MIN_CAPACITY = 1.0;
        private static final double SMOOTH = 0.8;
        private static final double BETA = 0.7;
        private static final double SCALE_CONSTANT = 0.4;
        private double fillRate;
        private double maxCapacity;
        private double currentCapacity;
        private boolean lastTimestampIsSet;
        private double lastTimestamp;
        private boolean enabled;
        private double measuredTxRate;
        private double lastTxRateBucket;
        private long requestCount;
        private double lastMaxRate;
        private double lastThrottleTime;
        private double timeWindow;
        private double newTokenBucketRate;

        private TransientState(PersistentState state) {
            this.fillRate = state.fillRate;
            this.maxCapacity = state.maxCapacity;
            this.currentCapacity = state.currentCapacity;
            this.lastTimestampIsSet = state.lastTimestampIsSet;
            this.lastTimestamp = state.lastTimestamp;
            this.enabled = state.enabled;
            this.measuredTxRate = state.measuredTxRate;
            this.lastTxRateBucket = state.lastTxRateBucket;
            this.requestCount = state.requestCount;
            this.lastMaxRate = state.lastMaxRate;
            this.lastThrottleTime = state.lastThrottleTime;
            this.timeWindow = state.timeWindow;
            this.newTokenBucketRate = state.newTokenBucketRate;
        }

        PersistentState toPersistent() {
            return new PersistentState(this);
        }

        Duration tokenBucketAcquire(RateLimiterClock clock, double amount) {
            if (!this.enabled) {
                return Duration.ZERO;
            }
            this.refill(clock);
            double waitTime = 0.0;
            if (this.currentCapacity < amount) {
                waitTime = (amount - this.currentCapacity) / this.fillRate;
            }
            this.currentCapacity -= amount;
            return Duration.ofNanos((long)(waitTime * 1.0E9));
        }

        void updateClientSendingRate(RateLimiterClock clock, boolean throttlingResponse) {
            double calculatedRate;
            this.updateMeasuredRate(clock);
            if (throttlingResponse) {
                double rateToUse = !this.enabled ? this.measuredTxRate : Math.min(this.measuredTxRate, this.fillRate);
                this.lastMaxRate = rateToUse;
                this.calculateTimeWindow();
                this.lastThrottleTime = clock.time();
                calculatedRate = this.cubicThrottle(rateToUse);
                this.enabled = true;
            } else {
                this.calculateTimeWindow();
                calculatedRate = this.cubicSuccess(clock.time());
            }
            double newRate = Math.min(calculatedRate, 2.0 * this.measuredTxRate);
            this.updateRate(clock, newRate);
        }

        void refill(RateLimiterClock clock) {
            double timestamp = clock.time();
            if (this.lastTimestampIsSet) {
                double fillAmount = (timestamp - this.lastTimestamp) * this.fillRate;
                this.currentCapacity = Math.min(this.maxCapacity, this.currentCapacity + fillAmount);
            }
            this.lastTimestamp = timestamp;
            this.lastTimestampIsSet = true;
        }

        void updateRate(RateLimiterClock clock, double newRps) {
            this.refill(clock);
            this.fillRate = Math.max(newRps, 0.5);
            this.maxCapacity = Math.max(newRps, 1.0);
            this.currentCapacity = Math.min(this.currentCapacity, this.maxCapacity);
            this.newTokenBucketRate = newRps;
        }

        void updateMeasuredRate(RateLimiterClock clock) {
            double time = clock.time();
            ++this.requestCount;
            double timeBucket = Math.floor(time * 2.0) / 2.0;
            if (timeBucket > this.lastTxRateBucket) {
                double currentRate = (double)this.requestCount / (timeBucket - this.lastTxRateBucket);
                this.measuredTxRate = currentRate * 0.8 + this.measuredTxRate * 0.19999999999999996;
                this.requestCount = 0L;
                this.lastTxRateBucket = timeBucket;
            }
        }

        void calculateTimeWindow() {
            this.timeWindow = Math.pow(this.lastMaxRate * 0.30000000000000004 / 0.4, 0.3333333333333333);
        }

        double cubicSuccess(double timestamp) {
            double delta = timestamp - this.lastThrottleTime;
            return 0.4 * Math.pow(delta - this.timeWindow, 3.0) + this.lastMaxRate;
        }

        double cubicThrottle(double rateToUse) {
            return rateToUse * 0.7;
        }
    }

    static class StateUpdate<T> {
        private final PersistentState newState;
        private final T result;

        StateUpdate(PersistentState newState, T result) {
            this.newState = newState;
            this.result = result;
        }
    }
}

