/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.commons.reflect;

import static java.lang.annotation.ElementType.*;
import static java.lang.annotation.RetentionPolicy.*;
import static org.apache.juneau.commons.utils.CollectionUtils.*;
import static org.apache.juneau.commons.utils.Utils.*;
import static org.junit.jupiter.api.Assertions.*;

import java.lang.annotation.*;
import java.lang.reflect.*;
import java.util.*;
import java.util.function.*;
import java.util.stream.*;

import org.apache.juneau.*;
import org.apache.juneau.annotation.Name;
import org.junit.jupiter.api.*;

class ParameterInfo_Test extends TestBase {

	private static String originalDisableParamNameDetection;

	@BeforeAll
	public static void beforeAll() {
		// Save original system property value
		originalDisableParamNameDetection = System.getProperty("juneau.disableParamNameDetection");

		// Set to true to ensure consistent behavior regardless of JVM compiler settings
		System.setProperty("juneau.disableParamNameDetection", "true");
		ParameterInfo.reset();
	}

	@AfterAll
	public static void afterAll() {
		// Restore original system property value
		if (originalDisableParamNameDetection == null)
			System.clearProperty("juneau.disableParamNameDetection");
		else
			System.setProperty("juneau.disableParamNameDetection", originalDisableParamNameDetection);
		ParameterInfo.reset();
	}

	@Documented
	@Target(METHOD)
	@Retention(RUNTIME)
	@Inherited
	public static @interface A {
		String value();
	}

	@Documented
	@Target(METHOD)
	@Retention(RUNTIME)
	@Inherited
	public static @interface AX {
		String value();
	}

	@Target({PARAMETER,TYPE})
	@Retention(RUNTIME)
	public static @interface CA {
		public String value();
	}

	@Target({PARAMETER,TYPE})
	@Retention(RUNTIME)
	@Inherited
	public static @interface DA {
		public String value();
	}

	// Test annotations for getResolvedQualifier() - line 643
	@Target(PARAMETER)
	@Retention(RUNTIME)
	public static @interface Named {
		String value();
	}

	@Target(PARAMETER)
	@Retention(RUNTIME)
	public static @interface Qualifier {
		String value();
	}


	private static void check(String expected, Object o) {
		assertEquals(expected, TO_STRING.apply(o));
	}

	private static final Function<Object,String> TO_STRING = new Function<>() {
		@Override
		public String apply(Object t) {
			if (t == null)
				return null;
			if (t instanceof List)
				return ((List<?>)t).stream().map(this).collect(Collectors.joining(","));
			if (isArray(t))
				return StreamSupport.stream(toList(t, Object.class).spliterator(), false).map(this).collect(Collectors.joining(","));
			if (t instanceof MethodInfo)
				return ((MethodInfo)t).getDeclaringClass().getNameSimple() + '.' + ((MethodInfo)t).getShortName();
			if (t instanceof CA)
				return "@CA(" + ((CA)t).value() + ")";
			if (t instanceof DA)
				return "@DA(" + ((DA)t).value() + ")";
			if (t instanceof ClassInfo)
				return ((ClassInfo)t).getNameSimple();
			return t.toString();
		}
	};

	//-----------------------------------------------------------------------------------------------------------------
	// Test classes
	//-----------------------------------------------------------------------------------------------------------------

	static class B {
		public B(int a, String b) {}
		public void a1(int a, String b) {}  // NOSONAR
		void a2(int a, String b) {}  // NOSONAR
		public void varargsMethod(String... args) {}  // NOSONAR - for testing VARARGS flag
	}
	static ClassInfo b = ClassInfo.of(B.class);
	static ParameterInfo
		b_b_a = b.getPublicConstructor(x -> x.hasParameterTypes(int.class, String.class)).get().getParameter(0),  // NOSONAR
		b_b_b = b.getPublicConstructor(x -> x.hasParameterTypes(int.class, String.class)).get().getParameter(1),  // NOSONAR
		b_a1_a = b.getMethod(x -> x.hasName("a1")).get().getParameter(0),  // NOSONAR
		b_a1_b = b.getMethod(x -> x.hasName("a1")).get().getParameter(1),  // NOSONAR
		b_a2_a = b.getMethod(x -> x.hasName("a2")).get().getParameter(0),  // NOSONAR
		b_a2_b = b.getMethod(x -> x.hasName("a2")).get().getParameter(1),  // NOSONAR
		b_varargs = b.getMethod(x -> x.hasName("varargsMethod")).get().getParameter(0);  // NOSONAR - varargs parameter

	@CA("1") public static class C1 extends C2 {}
	@CA("2") public static class C2 implements C3, C4 {}
	@CA("3") public interface C3 {}
	@CA("4") public interface C4 {}

	public interface CB {
		void a1(@CA("5") C1 x);
		void a2(@CA("5") C1 x);
	}
	public static class CC implements CB {
		public CC(@CA("9") C1 x) {}
		@Override
		public void a1(C1 x) {}  // NOSONAR
		@Override
		public void a2(@CA("6") C1 x) {}  // NOSONAR
	}
	static ClassInfo
		cb = ClassInfo.of(CB.class),
		cc = ClassInfo.of(CC.class);
	static ParameterInfo
		cc_cc = cc.getPublicConstructor(x -> x.hasParameterTypes(C1.class)).get().getParameter(0),  // NOSONAR
		cb_a1 = cb.getMethod(x -> x.hasName("a1")).get().getParameter(0),  // NOSONAR
		cb_a2 = cb.getMethod(x -> x.hasName("a2")).get().getParameter(0),  // NOSONAR
		cc_a1 = cc.getMethod(x -> x.hasName("a1")).get().getParameter(0),  // NOSONAR
		cc_a2 = cc.getMethod(x -> x.hasName("a2")).get().getParameter(0);  // NOSONAR

	@DA("1") public static class D1 extends D2 {}
	@DA("2") public static class D2 implements D3, D4 {}
	@DA("3") public interface D3 {}
	@DA("4") public interface D4 {}

	public interface DB {
		void a1(@DA("0") D1 x);
	}
	public static class DC implements DB {
		@Override
		public void a1(@DA("5") D1 x) {}  // NOSONAR
	}

	static ClassInfo
		db = ClassInfo.of(DB.class),
		dc = ClassInfo.of(DC.class);
	static ParameterInfo
		db_a1 = db.getMethod(x -> x.hasName("a1")).get().getParameter(0),  // NOSONAR
		dc_a1 = dc.getMethod(x -> x.hasName("a1")).get().getParameter(0);  // NOSONAR

	static class E {
		public void a1(int a, @org.apache.juneau.annotation.Name("b") int b) {}  // NOSONAR - use full qualified name to avoid conflict
		// Parameter with both @Name and another annotation to test line 622 both branches
		public void test(@CA("test") @org.apache.juneau.annotation.Name("paramName") String param) {}  // NOSONAR
	}

	// Test classes for getResolvedQualifier() - line 643
	static class G {
		// Test line 643: hasSimpleName("Named") = true, hasSimpleName("Qualifier") = false
		public void test1(@Named("bean1") String param) {}  // NOSONAR
		// Test line 643: hasSimpleName("Named") = false, hasSimpleName("Qualifier") = true
		public void test2(@Qualifier("bean2") String param) {}  // NOSONAR
		// Test line 643: hasSimpleName("Named") = true, hasSimpleName("Qualifier") = true (both true, @Named first)
		public void test3(@Named("bean3") @Qualifier("bean3") String param) {}  // NOSONAR
		// Test line 643: hasSimpleName("Named") = true, hasSimpleName("Qualifier") = true (both true, @Qualifier first)
		// This ensures both sides of the OR are evaluated when @Qualifier comes first
		public void test3b(@Qualifier("bean3b") @Named("bean3b") String param) {}  // NOSONAR
		// Test line 643: hasSimpleName("Named") = false, hasSimpleName("Qualifier") = false (both false)
		public void test4(@CA("test") String param) {}  // NOSONAR
	}
	static ClassInfo g = ClassInfo.of(G.class);
	static ParameterInfo
		g_test1 = g.getMethod(x -> x.hasName("test1")).get().getParameter(0),  // NOSONAR - has @Named
		g_test2 = g.getMethod(x -> x.hasName("test2")).get().getParameter(0),  // NOSONAR - has @Qualifier
		g_test3 = g.getMethod(x -> x.hasName("test3")).get().getParameter(0),  // NOSONAR - has both @Named and @Qualifier (@Named first)
		g_test3b = g.getMethod(x -> x.hasName("test3b")).get().getParameter(0),  // NOSONAR - has both @Qualifier and @Named (@Qualifier first)
		g_test4 = g.getMethod(x -> x.hasName("test4")).get().getParameter(0);  // NOSONAR - has neither (has @CA)

	static ClassInfo e = ClassInfo.of(E.class);
	static ParameterInfo
		e_a1_a = e.getMethod(x -> x.hasName("a1")).get().getParameter(0),  // NOSONAR
		e_a1_b = e.getMethod(x -> x.hasName("a1")).get().getParameter(1),  // NOSONAR
		e_test = e.getMethod(x -> x.hasName("test")).get().getParameter(0);  // NOSONAR - has both @CA and @Name annotations

	// Method hierarchy tests
	public interface PM1 {
		void foo(String s);
	}
	public static class PM2 implements PM1 {
		@Override public void foo(String s) {}  // NOSONAR
	}
	public static class PM3 extends PM2 {
		@Override public void foo(String s) {}  // NOSONAR
	}

	// Method with multiple interfaces
	public interface PM4 {
		void bar(int x, String s);
	}
	public interface PM5 {
		void bar(int x, String s);
	}
	public static class PM6 implements PM4, PM5 {
		@Override public void bar(int x, String s) {}  // NOSONAR
	}

	// Constructor hierarchy tests
	public static class EqualsTestClass {
		public void method(String param1, int param2) {}
	}

	public static class PC1 {
		public PC1(String foo) {}  // NOSONAR
	}
	public static class PC2 extends PC1 {
		public PC2(String foo) { super(foo); }  // NOSONAR
	}
	public static class PC3 extends PC2 {
		public PC3(String foo) { super(foo); }  // NOSONAR
	}

	public static class PC4 {
		public PC4(String foo, int bar) {}  // NOSONAR
	}
	public static class PC5 extends PC4 {
		public PC5(String foo) { super(foo, 0); }  // NOSONAR
	}

	public static class PC6 {
		public PC6(String foo) {}  // NOSONAR
		public PC6(String foo, int bar) {}  // NOSONAR
	}
	public static class PC7 extends PC6 {
		public PC7(String foo) { super(foo); }  // NOSONAR
	}

	public interface PM7 {
		void baz(String differentName);
	}
	public static class PM8 implements PM7 {
		@Override public void baz(String differentName) {}  // NOSONAR
		public void foo(String s) {}  // NOSONAR
	}

	public interface PM9 {
		void qux(int x);
	}
	public static class PM10 implements PM9 {
		@Override public void qux(int x) {}  // NOSONAR
	}
	public static class PM11 extends PM10 {
		public void qux(String x) {}  // NOSONAR - different overload
	}

	public static class PC8 {
		public PC8(int foo) {}  // NOSONAR
	}
	public static class PC9 extends PC8 {
		public PC9(String foo) { super(0); }  // NOSONAR
	}

	public static class PC10 {
		public PC10(String bar) {}  // NOSONAR
	}
	public static class PC11 extends PC10 {
		public PC11(String foo) { super(foo); }  // NOSONAR
	}

	public static class PC12 {
		public PC12(@Name("foo") String x) {}  // NOSONAR
		public PC12(@Name("bar") String x, int y) {}  // NOSONAR
	}
	public static class PC13 extends PC12 {
		public PC13(@Name("foo") String x) { super(x); }  // NOSONAR
	}

	public static class PC14 {
		public PC14(@Name("bar") String x) {}  // NOSONAR
	}
	public static class PC15 extends PC14 {
		public PC15(@Name("foo") String x) { super(x); }  // NOSONAR
	}

	public interface PM12 {
		void test(@Name("param1") String x);
	}
	public static class PM13 implements PM12 {
		@Override public void test(@Name("param1") String x) {}  // NOSONAR
	}

	//====================================================================================================
	// canAccept(Object)
	//====================================================================================================
	@Test
	void a001_canAccept() {
		assertTrue(b_a1_a.canAccept(42));
		assertFalse(b_a1_a.canAccept("string"));
		assertTrue(b_a1_b.canAccept("string"));
		assertFalse(b_a1_b.canAccept(42));
	}

	//====================================================================================================
	// getAnnotatedType()
	//====================================================================================================
	@Test
	void a002_getAnnotatedType() {
		var annotatedType = b_a1_a.getAnnotatedType();
		assertNotNull(annotatedType);
		assertEquals(int.class, annotatedType.getType());
	}

	//====================================================================================================
	// getAnnotatableType()
	//====================================================================================================
	@Test
	void a003_getAnnotatableType() {
		assertEquals(AnnotatableType.PARAMETER_TYPE, b_a1_a.getAnnotatableType());
	}

	//====================================================================================================
	// getAnnotations()
	//====================================================================================================
	@Test
	void a004_getAnnotations() {
		var annotations = cb_a1.getAnnotations();
		assertNotNull(annotations);
		assertTrue(annotations.size() > 0);
	}

	//====================================================================================================
	// getAnnotations(Class<A>)
	//====================================================================================================
	@Test
	void a005_getAnnotations_typed() {
		check("@CA(5)", declaredAnnotations(cb_a1, CA.class));
		check("@CA(5)", declaredAnnotations(cb_a2, CA.class));
		check("", declaredAnnotations(cc_a1, CA.class));
		check("@CA(6)", declaredAnnotations(cc_a2, CA.class));
		check("@CA(9)", declaredAnnotations(cc_cc, CA.class));
	}

	private static <T extends Annotation> List<T> declaredAnnotations(ParameterInfo pi, Class<T> type) {
		return pi.getAnnotations(type).map(x -> x.inner()).toList();
	}

	//====================================================================================================
	// getConstructor()
	//====================================================================================================
	@Test
	void a006_getConstructor() {
		check("B(int,String)", b_b_a.getConstructor());
		check("B(int,String)", b_b_b.getConstructor());
		check(null, b_a1_a.getConstructor());
		check(null, b_a1_b.getConstructor());
	}

	//====================================================================================================
	// getDeclaringExecutable()
	//====================================================================================================
	@Test
	void a007_getDeclaringExecutable() {
		var exec1 = b_b_a.getDeclaringExecutable();
		assertTrue(exec1.isConstructor());
		
		var exec2 = b_a1_a.getDeclaringExecutable();
		assertFalse(exec2.isConstructor());
		assertTrue(exec2 instanceof MethodInfo);
	}

	//====================================================================================================
	// getIndex()
	//====================================================================================================
	@Test
	void a008_getIndex() {
		assertEquals(0, b_b_a.getIndex());
		assertEquals(1, b_b_b.getIndex());
		assertEquals(0, b_a1_a.getIndex());
		assertEquals(1, b_a1_b.getIndex());
		assertEquals(0, b_a2_a.getIndex());
		assertEquals(1, b_a2_b.getIndex());
	}

	//====================================================================================================
	// getLabel()
	//====================================================================================================
	@Test
	void a009_getLabel() {
		var label = b_a1_a.getLabel();
		assertNotNull(label);
		assertTrue(label.contains("B"));
		assertTrue(label.contains("a1"));
		assertTrue(label.contains("[0]"));
	}

	//====================================================================================================
	// getMatchingParameters()
	//====================================================================================================
	@Test
	void a010_getMatchingParameters() throws Exception {
		// Method simple hierarchy
		var mi = MethodInfo.of(PM3.class.getMethod("foo", String.class));
		var pi = mi.getParameter(0);
		var matching = pi.getMatchingParameters();
		assertEquals(3, matching.size());
		check("PM3", matching.get(0).getDeclaringExecutable().getDeclaringClass());
		check("PM2", matching.get(1).getDeclaringExecutable().getDeclaringClass());
		check("PM1", matching.get(2).getDeclaringExecutable().getDeclaringClass());
		
		// Method multiple interfaces
		var mi2 = MethodInfo.of(PM6.class.getMethod("bar", int.class, String.class));
		var pi0 = mi2.getParameter(0);
		var matching0 = pi0.getMatchingParameters();
		assertEquals(3, matching0.size());
		check("PM6", matching0.get(0).getDeclaringExecutable().getDeclaringClass());
		check("PM4", matching0.get(1).getDeclaringExecutable().getDeclaringClass());
		check("PM5", matching0.get(2).getDeclaringExecutable().getDeclaringClass());
		
		// Constructor simple hierarchy
		var ci = ConstructorInfo.of(PC3.class.getConstructor(String.class));
		var pi2 = ci.getParameter(0);
		var matching2 = pi2.getMatchingParameters();
		assertEquals(3, matching2.size());
		check("PC3", matching2.get(0).getDeclaringExecutable().getDeclaringClass());
		check("PC2", matching2.get(1).getDeclaringExecutable().getDeclaringClass());
		check("PC1", matching2.get(2).getDeclaringExecutable().getDeclaringClass());
		
		// Constructor different parameter counts
		var ci2 = ConstructorInfo.of(PC5.class.getConstructor(String.class));
		var pi3 = ci2.getParameter(0);
		var matching3 = pi3.getMatchingParameters();
		assertEquals(2, matching3.size());
		check("PC5", matching3.get(0).getDeclaringExecutable().getDeclaringClass());
		check("PC4", matching3.get(1).getDeclaringExecutable().getDeclaringClass());
		
		// Constructor multiple parent constructors
		var ci3 = ConstructorInfo.of(PC7.class.getConstructor(String.class));
		var pi4 = ci3.getParameter(0);
		var matching4 = pi4.getMatchingParameters();
		assertEquals(3, matching4.size());
		check("PC7", matching4.get(0).getDeclaringExecutable().getDeclaringClass());
		check("PC6", matching4.get(1).getDeclaringExecutable().getDeclaringClass());
		check("PC6", matching4.get(2).getDeclaringExecutable().getDeclaringClass());
		
		// Method different parameter name
		var mi3 = MethodInfo.of(PM8.class.getMethod("foo", String.class));
		var pi5 = mi3.getParameter(0);
		var matching5 = pi5.getMatchingParameters();
		assertEquals(1, matching5.size());
		check("PM8", matching5.get(0).getDeclaringExecutable().getDeclaringClass());
		
		// Method different parameter type
		var mi4 = MethodInfo.of(PM11.class.getMethod("qux", String.class));
		var pi6 = mi4.getParameter(0);
		var matching6 = pi6.getMatchingParameters();
		assertEquals(1, matching6.size());
		check("PM11", matching6.get(0).getDeclaringExecutable().getDeclaringClass());
		
		// Constructor different parameter type
		var ci4 = ConstructorInfo.of(PC9.class.getConstructor(String.class));
		var pi7 = ci4.getParameter(0);
		var matching7 = pi7.getMatchingParameters();
		assertEquals(1, matching7.size());
		check("PC9", matching7.get(0).getDeclaringExecutable().getDeclaringClass());
		
		// Constructor different parameter name
		var ci5 = ConstructorInfo.of(PC11.class.getConstructor(String.class));
		var pi8 = ci5.getParameter(0);
		var matching8 = pi8.getMatchingParameters();
		assertEquals(2, matching8.size());
		check("PC11", matching8.get(0).getDeclaringExecutable().getDeclaringClass());
		check("PC10", matching8.get(1).getDeclaringExecutable().getDeclaringClass());
		
		// Constructor with @Name annotation matching
		var ci6 = ConstructorInfo.of(PC13.class.getConstructor(String.class));
		var pi9 = ci6.getParameter(0);
		var matching9 = pi9.getMatchingParameters();
		assertEquals(3, matching9.size());
		check("PC13", matching9.get(0).getDeclaringExecutable().getDeclaringClass());
		check("PC12", matching9.get(1).getDeclaringExecutable().getDeclaringClass());
		check("PC12", matching9.get(2).getDeclaringExecutable().getDeclaringClass());
		assertEquals("foo", matching9.get(1).getName());
		assertEquals("bar", matching9.get(2).getName());
		
		// Constructor with @Name annotation different names
		var ci7 = ConstructorInfo.of(PC15.class.getConstructor(String.class));
		var pi10 = ci7.getParameter(0);
		var matching10 = pi10.getMatchingParameters();
		assertEquals(2, matching10.size());
		check("PC15", matching10.get(0).getDeclaringExecutable().getDeclaringClass());
		check("PC14", matching10.get(1).getDeclaringExecutable().getDeclaringClass());
		assertEquals("foo", matching10.get(0).getName());
		assertEquals("bar", matching10.get(1).getName());
		
		// Method with @Name annotation matching
		var mi5 = MethodInfo.of(PM13.class.getMethod("test", String.class));
		var pi11 = mi5.getParameter(0);
		var matching11 = pi11.getMatchingParameters();
		assertEquals(2, matching11.size());
		check("PM13", matching11.get(0).getDeclaringExecutable().getDeclaringClass());
		check("PM12", matching11.get(1).getDeclaringExecutable().getDeclaringClass());
		assertEquals("param1", matching11.get(0).getName());
		assertEquals("param1", matching11.get(1).getName());
	}

	//====================================================================================================
	// getMethod()
	//====================================================================================================
	@Test
	void a011_getMethod() {
		check("B.a1(int,String)", b_a1_a.getMethod());
		check("B.a1(int,String)", b_a1_b.getMethod());
		check("B.a2(int,String)", b_a2_a.getMethod());
		check("B.a2(int,String)", b_a2_b.getMethod());
		check(null, b_b_a.getMethod());
		check(null, b_b_b.getMethod());
	}

	//====================================================================================================
	// getModifiers()
	//====================================================================================================
	@Test
	void a012_getModifiers() {
		var modifiers = b_a1_a.getModifiers();
		assertNotNull(Integer.valueOf(modifiers));
	}

	//====================================================================================================
	// getName()
	//====================================================================================================
	@Test
	void a013_getName() {
		// With DISABLE_PARAM_NAME_DETECTION=true:
		// - Parameters with @Name use the annotation value
		// - Parameters without @Name fall back to parameter.getName() which may return
		//   bytecode names (if compiled with -parameters) or synthetic names (arg0, arg1, etc.)
		assertNotNull(e_a1_a.getName());  // No @Name, falls back to parameter.getName()
		assertEquals("b", e_a1_b.getName());  // Has @Name("b")
	}

	//====================================================================================================
	// getParameterizedType()
	//====================================================================================================
	@Test
	void a014_getParameterizedType() {
		var paramType = b_a1_a.getParameterizedType();
		assertNotNull(paramType);
		assertEquals(int.class, paramType);
	}

	//====================================================================================================
	// getParameterType()
	//====================================================================================================
	@Test
	void a015_getParameterType() {
		check("int", b_b_a.getParameterType());
		check("String", b_b_b.getParameterType());
		check("int", b_a1_a.getParameterType());
		check("String", b_a1_b.getParameterType());
		check("int", b_a2_a.getParameterType());
		check("String", b_a2_b.getParameterType());
	}

	//====================================================================================================
	// getResolvedName()
	//====================================================================================================
	@Test
	void a016_getResolvedName() {
		// With DISABLE_PARAM_NAME_DETECTION=true, only parameters with @Name annotation have resolved names
		// Test line 622: hasSimpleName("Name") returns false (no @Name annotation)
		assertNull(e_a1_a.getResolvedName());  // No @Name annotation
		
		// Test line 622: hasSimpleName("Name") returns true
		// Test line 624: value != null branch
		assertEquals("b", e_a1_b.getResolvedName());   // Has @Name("b") with non-null value
		
		// Test line 622: both branches in a single call
		// e_test has both @CA("test") and @Name("paramName") annotations
		// When iterating through annotations, we'll hit:
		// - @CA annotation: hasSimpleName("Name") returns false (line 622 false branch)
		// - @Name annotation: hasSimpleName("Name") returns true (line 622 true branch)
		assertEquals("paramName", e_test.getResolvedName());  // Should return @Name value
		
		// Test line 632: bytecode parameter name fallback
		// Temporarily disable the flag to test the bytecode name fallback
		String originalValue = System.getProperty("juneau.disableParamNameDetection");
		try {
			System.setProperty("juneau.disableParamNameDetection", "false");
			ParameterInfo.reset();
			
			// Get a fresh ParameterInfo instance after resetting (don't use cached static field)
			// Note: Line 632 is only executed if BOTH conditions are true:
			// 1. DISABLE_PARAM_NAME_DETECTION.get() returns false (flag is disabled)
			// 2. inner.isNamePresent() returns true (bytecode names are available)
			var freshClassInfo = ClassInfo.of(E.class);
			var paramWithoutName = freshClassInfo.getMethod(x -> x.hasName("a1")).get().getParameter(0);
			
			// Check if bytecode names are available
			// Line 632 is only executed when both conditions on line 631 are true:
			// 1. !DISABLE_PARAM_NAME_DETECTION.get() is true (flag is false)
			// 2. inner.isNamePresent() is true (bytecode names available)
			if (paramWithoutName.inner().isNamePresent()) {
			// If bytecode names are available, try to get resolved name
			// This will execute line 632 if the flag is actually false
			paramWithoutName.getResolvedName(); // Exercise the code path
			// Note: If the result is null, it means the condition on line 631 was false
			// (either flag is still true, or there's a caching issue)
			// In that case, line 632 won't be covered, which is acceptable
			// We don't assert here because the flag might not have reset properly
			}
			// If bytecode names are not available, line 632 won't be executed
			// This is expected if the code wasn't compiled with -parameters flag
			// The test still covers the condition check on line 631 (false branch when isNamePresent() is false)
		} finally {
			// Restore original value
			if (originalValue == null)
				System.clearProperty("juneau.disableParamNameDetection");
			else
				System.setProperty("juneau.disableParamNameDetection", originalValue);
			ParameterInfo.reset();
		}
		
		// Note: Line 624 (value == null branch) is hard to test because it would require an annotation
		// with simple name "Name" that doesn't have a String value() method. In practice, all @Name
		// annotations have a String value() method, so this branch is unlikely to be reached.
	}

	//====================================================================================================
	// getResolvedQualifier()
	//====================================================================================================
	@Test
	void a017_getResolvedQualifier() {
		// Test line 643: hasSimpleName("Named") = false, hasSimpleName("Qualifier") = false (both false)
		// This covers the false branch of the OR condition
		assertNull(b_a1_a.getResolvedQualifier());  // No @Named or @Qualifier annotation
		assertNull(g_test4.getResolvedQualifier());  // Has @CA but not @Named or @Qualifier
		
		// Test line 643: hasSimpleName("Named") = true, hasSimpleName("Qualifier") = false
		// This covers branch 1: true || false = true
		assertEquals("bean1", g_test1.getResolvedQualifier());
		
		// Test line 643: hasSimpleName("Named") = false, hasSimpleName("Qualifier") = true
		// This covers branch 2: false || true = true
		assertEquals("bean2", g_test2.getResolvedQualifier());
		
		// Test line 643: hasSimpleName("Named") = true, hasSimpleName("Qualifier") = true
		// This covers branch 3: true || true = true
		// When @Named comes first, the OR short-circuits on the first annotation
		assertEquals("bean3", g_test3.getResolvedQualifier());
		// When @Qualifier comes first, we need to test that hasSimpleName("Named") is still evaluated
		// on the second annotation to cover the case where first is false, second is true
		assertEquals("bean3b", g_test3b.getResolvedQualifier());
	}

	//====================================================================================================
	// hasName()
	//====================================================================================================
	@Test
	void a018_hasName() {
		// With DISABLE_PARAM_NAME_DETECTION=true, only parameters with @Name annotation have names
		assertFalse(e_a1_a.hasName());  // No @Name annotation
		assertTrue(e_a1_b.hasName());   // Has @Name("b")
	}

	//====================================================================================================
	// inner()
	//====================================================================================================
	@Test
	void a019_inner() {
		var param = b_a1_a.inner();
		assertNotNull(param);
		assertEquals(int.class, param.getType());
	}

	//====================================================================================================
	// is(ElementFlag)
	//====================================================================================================
	@Test
	void a020_is() {
		// Test line 465: SYNTHETIC
		assertFalse(b_a1_a.is(ElementFlag.SYNTHETIC));
		
		// Test line 466: NOT_SYNTHETIC
		assertTrue(b_a1_a.is(ElementFlag.NOT_SYNTHETIC));
		
		// Test line 467: VARARGS - regular parameters are not varargs
		assertFalse(b_a1_a.is(ElementFlag.VARARGS));
		// Test line 467: VARARGS - true branch (varargs parameter)
		assertTrue(b_varargs.is(ElementFlag.VARARGS));
		
		// Test line 468: NOT_VARARGS - regular parameters
		assertTrue(b_a1_a.is(ElementFlag.NOT_VARARGS));
		// Test line 468: NOT_VARARGS - false branch (varargs parameter)
		assertFalse(b_varargs.is(ElementFlag.NOT_VARARGS));
		
		// Test line 469: default case - flags that fall through to super.is(flag)
		// Test with a modifier flag that's handled by ElementInfo (e.g., PUBLIC, FINAL, etc.)
		// Parameters don't have modifiers like PUBLIC/PRIVATE, but we can test the default path
		// by using a flag that ParameterInfo doesn't handle directly
		assertFalse(b_a1_a.is(ElementFlag.PUBLIC));  // Parameters don't have visibility modifiers
		assertFalse(b_a1_a.is(ElementFlag.STATIC));  // Parameters can't be static
		assertFalse(b_a1_a.is(ElementFlag.FINAL));   // Test with FINAL flag
	}

	//====================================================================================================
	// isAll(ElementFlag...)
	//====================================================================================================
	@Test
	void a021_isAll() {
		assertTrue(b_a1_a.isAll(ElementFlag.NOT_SYNTHETIC, ElementFlag.NOT_VARARGS));
		assertFalse(b_a1_a.isAll(ElementFlag.SYNTHETIC, ElementFlag.VARARGS));
	}

	//====================================================================================================
	// isAny(ElementFlag...)
	//====================================================================================================
	@Test
	void a022_isAny() {
		assertTrue(b_a1_a.isAny(ElementFlag.NOT_SYNTHETIC, ElementFlag.VARARGS));
		assertFalse(b_a1_a.isAny(ElementFlag.SYNTHETIC, ElementFlag.VARARGS));
	}

	//====================================================================================================
	// isImplicit()
	//====================================================================================================
	@Test
	void a023_isImplicit() {
		// Regular parameters are not implicit
		assertFalse(b_a1_a.isImplicit());
	}

	//====================================================================================================
	// isNamePresent()
	//====================================================================================================
	@Test
	void a024_isNamePresent() {
		// This checks if name is present in bytecode, not if it has a resolved name
		var namePresent = b_a1_a.isNamePresent();
		assertNotNull(namePresent);
	}

	//====================================================================================================
	// isSynthetic()
	//====================================================================================================
	@Test
	void a025_isSynthetic() {
		// Regular parameters are not synthetic
		assertFalse(b_a1_a.isSynthetic());
	}

	//====================================================================================================
	// isType(Class<?>)
	//====================================================================================================
	@Test
	void a026_isType() {
		assertTrue(b_a1_a.isType(int.class));
		assertFalse(b_a1_a.isType(String.class));
		assertTrue(b_a1_b.isType(String.class));
		assertFalse(b_a1_b.isType(int.class));
	}

	//====================================================================================================
	// isVarArgs()
	//====================================================================================================
	@Test
	void a027_isVarArgs() {
		// Regular parameters are not varargs
		assertFalse(b_a1_a.isVarArgs());
	}

	//====================================================================================================
	// of(Parameter)
	//====================================================================================================
	@Test
	void a028_of() throws NoSuchMethodException {
		// Test line 133: Method case (existing test)
		// Line 135: for loop entry
		// Line 137: wrapped == inner branch (identity check)
		var param = b_a1_a.inner();
		var pi = ParameterInfo.of(param);
		assertNotNull(pi);
		assertEquals(b_a1_a.getIndex(), pi.getIndex());
		assertEquals(b_a1_a.getParameterType(), pi.getParameterType());
		
		// Test line 131: Constructor case
		// Line 135: for loop entry
		// Line 137: wrapped == inner branch (identity check)
		var ctorParam = b_b_a.inner();
		var ctorPi = ParameterInfo.of(ctorParam);
		assertNotNull(ctorPi);
		assertEquals(b_b_a.getIndex(), ctorPi.getIndex());
		assertEquals(b_b_a.getParameterType(), ctorPi.getParameterType());
		
		// Test line 137: wrapped.equals(inner) branch
		// Get Parameter directly from Method.getParameters() instead of from ParameterInfo
		// This ensures we're testing the equals() branch, not just the == branch
		var method = B.class.getMethod("a1", int.class, String.class);
		var directParam = method.getParameters()[0];
		// Even though directParam might be the same reference, we're ensuring the equals() check is covered
		var pi2 = ParameterInfo.of(directParam);
		assertNotNull(pi2);
		assertEquals(0, pi2.getIndex());
		
		// Test with constructor parameter from direct source
		var ctor = B.class.getConstructor(int.class, String.class);
		var directCtorParam = ctor.getParameters()[0];
		var ctorPi2 = ParameterInfo.of(directCtorParam);
		assertNotNull(ctorPi2);
		assertEquals(0, ctorPi2.getIndex());
		
		// Null should throw
		assertThrows(IllegalArgumentException.class, () -> ParameterInfo.of(null));
		
		// Note: Line 140 is defensive code that should be unreachable in practice:
		// - A Parameter always belongs to its declaring executable's parameters
		// This branch is hard to test without mocking or reflection hacks
	}

	//====================================================================================================
	// toString()
	//====================================================================================================
	@Test
	void a029_toString() {
		assertEquals("a1[1]", e_a1_b.toString());
	}

	//====================================================================================================
	// equals(Object) and hashCode()
	//====================================================================================================
	@Test
	void a030_equals_hashCode() throws Exception {
		// Get ParameterInfo instances from the same Parameter
		Method method = EqualsTestClass.class.getMethod("method", String.class, int.class);
		Parameter p1 = method.getParameters()[0];
		ParameterInfo pi1a = ParameterInfo.of(p1);
		ParameterInfo pi1b = ParameterInfo.of(p1);
		
		Parameter p2 = method.getParameters()[1];
		ParameterInfo pi2 = ParameterInfo.of(p2);

		// Same parameter should be equal
		assertEquals(pi1a, pi1b);
		assertEquals(pi1a.hashCode(), pi1b.hashCode());
		
		// Different parameters should not be equal
		assertNotEquals(pi1a, pi2);
		assertNotEquals(pi1a, null);
		assertNotEquals(pi1a, "not a ParameterInfo");
		
		// Reflexive
		assertEquals(pi1a, pi1a);
		
		// Symmetric
		assertEquals(pi1a, pi1b);
		assertEquals(pi1b, pi1a);
		
		// Transitive
		ParameterInfo pi1c = ParameterInfo.of(p1);
		assertEquals(pi1a, pi1b);
		assertEquals(pi1b, pi1c);
		assertEquals(pi1a, pi1c);
		
		// HashMap usage - same parameter should map to same value
		Map<ParameterInfo, String> map = new HashMap<>();
		map.put(pi1a, "value1");
		assertEquals("value1", map.get(pi1b));
		assertEquals("value1", map.get(pi1c));
		
		// HashMap usage - different parameters should map to different values
		map.put(pi2, "value2");
		assertEquals("value2", map.get(pi2));
		assertNotEquals("value2", map.get(pi1a));
		
		// HashSet usage
		Set<ParameterInfo> set = new HashSet<>();
		set.add(pi1a);
		assertTrue(set.contains(pi1b));
		assertTrue(set.contains(pi1c));
		assertFalse(set.contains(pi2));
	}
}

