/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tomcat.jakartaee;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.tomcat.jakartaee.CacheEntry;
import org.apache.tomcat.jakartaee.EESpecProfile;
import org.apache.tomcat.jakartaee.StringManager;

public class MigrationCache {
    private static final Logger logger = Logger.getLogger(MigrationCache.class.getCanonicalName());
    private static final StringManager sm = StringManager.getManager(MigrationCache.class);
    private static final String METADATA_FILE = "cache-metadata.txt";
    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ISO_LOCAL_DATE;
    private final File cacheDir;
    private final int retentionDays;
    private final Map<String, LocalDate> cacheMetadata;
    private final File metadataFile;

    public MigrationCache(File cacheDir, int retentionDays) throws IOException {
        this.retentionDays = retentionDays;
        this.cacheMetadata = new HashMap<String, LocalDate>();
        this.cacheDir = cacheDir;
        File file = this.metadataFile = cacheDir == null ? null : new File(cacheDir, METADATA_FILE);
        if (cacheDir == null) {
            throw new IllegalStateException(sm.getString("cache.nullDirectory"));
        }
        if (!cacheDir.exists() && !cacheDir.mkdirs()) {
            throw new IOException(sm.getString("cache.cannotCreate", cacheDir.getAbsolutePath()));
        }
        if (!cacheDir.isDirectory()) {
            throw new IOException(sm.getString("cache.notDirectory", cacheDir.getAbsolutePath()));
        }
        this.loadMetadata();
        this.cleanupTempFiles();
        logger.log(Level.INFO, sm.getString("cache.enabled", cacheDir.getAbsolutePath(), retentionDays));
    }

    private void cleanupTempFiles() {
        File[] files = this.cacheDir.listFiles();
        if (files != null) {
            int cleanedCount = 0;
            for (File file : files) {
                if (!file.isFile() || !file.getName().startsWith("temp-") || !file.getName().endsWith(".tmp")) continue;
                if (file.delete()) {
                    ++cleanedCount;
                    logger.log(Level.FINE, sm.getString("cache.tempfile.cleaned", file.getName()));
                    continue;
                }
                logger.log(Level.WARNING, sm.getString("cache.tempfile.cleanFailed", file.getName()));
            }
            if (cleanedCount > 0) {
                logger.log(Level.INFO, sm.getString("cache.tempfiles.cleaned", cleanedCount));
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private void loadMetadata() {
        LocalDate today = LocalDate.now();
        if (!this.metadataFile.exists()) {
            logger.log(Level.FINE, sm.getString("cache.metadata.notFound"));
            this.scanCacheDirectory(today);
            return;
        }
        try (BufferedReader reader = new BufferedReader(new FileReader(this.metadataFile));){
            String line;
            while ((line = reader.readLine()) != null) {
                if ((line = line.trim()).isEmpty() || line.startsWith("#")) continue;
                String[] parts = line.split("\\|");
                if (parts.length == 2) {
                    String hash = parts[0];
                    try {
                        LocalDate lastAccessed = LocalDate.parse(parts[1], DATE_FORMATTER);
                        this.cacheMetadata.put(hash, lastAccessed);
                        continue;
                    }
                    catch (DateTimeParseException e) {
                        logger.log(Level.WARNING, sm.getString("cache.metadata.invalidDate", line));
                        continue;
                    }
                }
                logger.log(Level.WARNING, sm.getString("cache.metadata.invalidLine", line));
            }
            Set<String> existingHashes = this.scanCacheDirectory(null);
            for (String hash : existingHashes) {
                if (this.cacheMetadata.containsKey(hash)) continue;
                this.cacheMetadata.put(hash, today);
            }
            logger.log(Level.FINE, sm.getString("cache.metadata.loaded", this.cacheMetadata.size()));
            return;
        }
        catch (IOException e) {
            logger.log(Level.WARNING, sm.getString("cache.metadata.loadError"), e);
            this.cacheMetadata.clear();
            this.scanCacheDirectory(today);
        }
    }

    private Set<String> scanCacheDirectory(LocalDate accessDate) {
        HashSet<String> hashes = new HashSet<String>();
        File[] subdirs = this.cacheDir.listFiles();
        if (subdirs != null) {
            for (File subdir : subdirs) {
                File[] files;
                if (!subdir.isDirectory() || (files = subdir.listFiles()) == null) continue;
                for (File file : files) {
                    if (!file.isFile() || !file.getName().endsWith(".jar")) continue;
                    String hash = file.getName().substring(0, file.getName().length() - 4);
                    hashes.add(hash);
                    if (accessDate == null) continue;
                    this.cacheMetadata.put(hash, accessDate);
                }
            }
        }
        return hashes;
    }

    public CacheEntry getCacheEntry(byte[] sourceBytes, EESpecProfile profile) throws IOException {
        String hash = this.computeHash(sourceBytes, profile);
        File cachedFile = this.getCacheFile(hash);
        boolean exists = cachedFile.exists();
        File tempFile = new File(this.cacheDir, "temp-" + UUID.randomUUID() + ".tmp");
        this.updateAccessTime(hash);
        return new CacheEntry(hash, exists, cachedFile, tempFile);
    }

    private File getCacheFile(String hash) {
        String subdir = hash.substring(0, 2);
        File subdirFile = new File(this.cacheDir, subdir);
        if (!subdirFile.exists()) {
            subdirFile.mkdirs();
        }
        return new File(subdirFile, hash + ".jar");
    }

    private String computeHash(byte[] bytes, EESpecProfile profile) throws IOException {
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            digest.update(profile.toString().getBytes(StandardCharsets.UTF_8));
            digest.update(bytes);
            byte[] hashBytes = digest.digest();
            StringBuilder sb = new StringBuilder();
            for (byte b : hashBytes) {
                sb.append(String.format("%02x", b));
            }
            return sb.toString();
        }
        catch (NoSuchAlgorithmException e) {
            throw new IOException(sm.getString("cache.hashError"), e);
        }
    }

    public void clear() throws IOException {
        this.deleteDirectory(this.cacheDir);
        this.cacheDir.mkdirs();
        logger.log(Level.INFO, sm.getString("cache.cleared"));
    }

    private void deleteDirectory(File dir) throws IOException {
        File[] files;
        if (dir.isDirectory() && (files = dir.listFiles()) != null) {
            for (File file : files) {
                this.deleteDirectory(file);
            }
        }
        if (!Files.deleteIfExists(dir.toPath()) && dir.exists()) {
            throw new IOException(sm.getString("cache.deleteFailed", dir.getAbsolutePath()));
        }
    }

    private void updateAccessTime(String hash) {
        this.cacheMetadata.put(hash, LocalDate.now());
    }

    private void saveMetadata() throws IOException {
        try (BufferedWriter writer = new BufferedWriter(new FileWriter(this.metadataFile));){
            writer.write("# Migration cache metadata - hash|last_access_date\n");
            for (Map.Entry<String, LocalDate> entry : this.cacheMetadata.entrySet()) {
                writer.write(entry.getKey());
                writer.write("|");
                writer.write(entry.getValue().format(DATE_FORMATTER));
                writer.write("\n");
            }
        }
        logger.log(Level.FINE, sm.getString("cache.metadata.saved", this.cacheMetadata.size()));
    }

    public void pruneCache() throws IOException {
        LocalDate cutoffDate = LocalDate.now().minusDays(this.retentionDays);
        int prunedCount = 0;
        long prunedSize = 0L;
        HashSet<String> toRemove = new HashSet<String>();
        for (Map.Entry<String, LocalDate> entry : this.cacheMetadata.entrySet()) {
            String hash = entry.getKey();
            LocalDate lastAccessed = entry.getValue();
            if (!lastAccessed.isBefore(cutoffDate)) continue;
            File cachedFile = this.getCacheFile(hash);
            if (cachedFile.exists()) {
                long fileSize = cachedFile.length();
                if (cachedFile.delete()) {
                    prunedSize += fileSize;
                    ++prunedCount;
                    toRemove.add(hash);
                    logger.log(Level.FINE, sm.getString("cache.pruned.entry", hash, lastAccessed));
                    continue;
                }
                logger.log(Level.WARNING, sm.getString("cache.pruned.failed", hash));
                continue;
            }
            toRemove.add(hash);
        }
        for (String hash : toRemove) {
            this.cacheMetadata.remove(hash);
        }
        this.saveMetadata();
        if (prunedCount > 0) {
            logger.log(Level.INFO, sm.getString("cache.pruned.summary", prunedCount, prunedSize / 1024L / 1024L, this.retentionDays));
        } else {
            logger.log(Level.FINE, sm.getString("cache.pruned.none", this.retentionDays));
        }
    }

    public void finalizeCacheOperations() throws IOException {
        this.saveMetadata();
        this.pruneCache();
    }

    public String getStats() {
        long totalSize = 0L;
        int entryCount = 0;
        File[] subdirs = this.cacheDir.listFiles();
        if (subdirs != null) {
            for (File subdir : subdirs) {
                File[] files;
                if (!subdir.isDirectory() || (files = subdir.listFiles()) == null) continue;
                for (File file : files) {
                    if (!file.isFile()) continue;
                    totalSize += file.length();
                    ++entryCount;
                }
            }
        }
        return sm.getString("cache.stats", entryCount, totalSize / 1024L / 1024L);
    }
}

